/*
 * This file is part of MiToBo, the Microscope Image Analysis Toolbox.
 *
 * Copyright (C) 2010 - 2025
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Fore more information on MiToBo, visit
 *
 *    http://www.informatik.uni-halle.de/mitobo/
 *
 */

package de.unihalle.informatik.MiToBo.color.tools;

import de.unihalle.informatik.Alida.exceptions.ALDOperatorException;
import de.unihalle.informatik.Alida.exceptions.ALDProcessingDAGException;

import java.awt.Color;

import de.unihalle.informatik.Alida.annotations.ALDAOperator;
import de.unihalle.informatik.Alida.annotations.ALDAOperator.Level;
import de.unihalle.informatik.Alida.annotations.Parameter;
import de.unihalle.informatik.MiToBo.color.conversion.HSVToRGBPixelConverter;
import de.unihalle.informatik.MiToBo.core.operator.*;

/**
 * Operator generating a list of N colors with subsequent colors being
 * significantly different to each other.
 * <p>
 * The list is generated by sampling from the hue color circle. The circle is
 * partitioned by equally sized sampling points according to the number of
 * colors the list should finally contain. Each color is transformed to RGB
 * color space and added to the list.
 * <p>
 * If the number of colors requested is even the circle is directly sampled
 * with the given number, if the number is odd (for simplicity of sampling) the
 * number is internally increased by one, i.e., an even number of partitions is
 * generated, and the first N-1 colors are sampled and returned in the list.
 * <p>
 * For ensuring maximal differences between subsequent colors the list is
 * filled as follows. We start with a hue angle of zero (red) and traverse the
 * sampling points along the circle in order of increasing hue angles from 0
 * to 180 degrees. Between each two sampling points p1 and p2 the color of the
 * sampling point on the circle opposit to p1 is inserted into the list which
 * is most different from p1 and p2.
 *
 * @author moeller
 */
@ALDAOperator(genericExecutionMode=ALDAOperator.ExecutionMode.NONE,
		level=Level.STANDARD)
public class DistinctColorListGenerator extends MTBOperator {

	/**
	 * Number of colors to generate.
	 */
	@Parameter( label= "Number of Colors", required = true, dataIOOrder = 1,
		direction = Parameter.Direction.IN, description = "Number of colors.")
	protected int nColors = 8;

	/**
	 * Color list.
	 */
	@Parameter( label= "Resulting Color List",
		direction = Parameter.Direction.OUT, description = "Color list.")
	private transient Color[] resultList = null;

	/**
	 * Default constructor.
	 *  @throws ALDOperatorException
	 */
	public DistinctColorListGenerator() throws ALDOperatorException {
		// nothing to do here
	}

	/**
	 * Returns color list.
	 */
	public Color[] getColorList() {
		return this.resultList;
	}

	/**
	 * Set number of colors.
	 * @param n	Number of colors.
	 */
	public void setColorNumber(int n) {
		this.nColors = n;
	}

	/**
	 * This method does the actual work.
	 * @throws ALDOperatorException
	 * @throws ALDProcessingDAGException
	 */
	@Override
	protected void operate()
			throws ALDOperatorException, ALDProcessingDAGException {

		double angleStep = 360.0/(double)this.nColors;
		this.resultList = new Color[this.nColors];

		int evenN = this.nColors;
		if (evenN%2 != 0)
			++evenN;

		HSVToRGBPixelConverter pixConv;
		int rgb[];
		double h;
		for (int i=0; i<evenN/2; ++i) {

			h = i * angleStep;
			pixConv = new HSVToRGBPixelConverter(h, 1.0, 1.0);
			pixConv.runOp();
			rgb = pixConv.getRGBResult();
			this.resultList[2*i] = new Color(rgb[0], rgb[1], rgb[2]);

			// stop if end of array is reached
			if (2*i+1 >= this.nColors)
				break;

			// opposite color
			h = (i + evenN/2) % evenN * angleStep;
			pixConv = new HSVToRGBPixelConverter(h, 1.0, 1.0);
			pixConv.runOp();
			rgb = pixConv.getRGBResult();
			this.resultList[2*i + 1] = new Color(rgb[0], rgb[1], rgb[2]);
		}
	}
}
