/*
 * This file is part of MiToBo, the Microscope Image Analysis Toolbox.
 *
 * Copyright (C) 2010
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Fore more information on MiToBo, visit
 *
 *    http://www.informatik.uni-halle.de/mitobo/
 *
 */

package de.unihalle.informatik.MiToBo.core.imageJ;

import java.awt.geom.Point2D;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;

import de.unihalle.informatik.Alida.annotations.ALDAOperator;
import de.unihalle.informatik.Alida.annotations.ALDAOperator.Level;
import de.unihalle.informatik.Alida.annotations.Parameter;
import de.unihalle.informatik.Alida.annotations.Parameter.Direction;
import de.unihalle.informatik.Alida.datatypes.ALDFileString;
import de.unihalle.informatik.Alida.exceptions.ALDOperatorException;
import de.unihalle.informatik.Alida.exceptions.ALDOperatorException.OperatorExceptionType;
import de.unihalle.informatik.Alida.exceptions.ALDProcessingDAGException;
import de.unihalle.informatik.Alida.helpers.ALDFilePathManipulator;
import de.unihalle.informatik.Alida.operator.events.ALDOperatorExecutionProgressEvent;
import de.unihalle.informatik.MiToBo.core.datatypes.*;
import de.unihalle.informatik.MiToBo.core.operator.MTBOperator;

/**
 * Operator to convert contours of ImageJ ROIs to CSV format.
 * <p>
 * Each ROI in the provided input file is converted into a CSV output file. 
 * In detail, the CSV file will contain the coordinates of the contour points
 * of the ROI in integer format. Each row contains a single point where x and
 * y coordinates are separated by a comma.
 * <p>
 * For files with ending ".roi" containing a single ROI also a single csv file
 * with the given output filename is written. In case of a ROI zip archive 
 * containing multiple ROIs for each ROI an individual csv file is written.
 * The names of these files are generated by extracting the root of the given
 * output file name, adding a unique number and finally appending ending
 * ".csv" to the filename.
 * 
 * @author moeller
 */
@ALDAOperator(genericExecutionMode=ALDAOperator.ExecutionMode.ALL, 
	level=Level.STANDARD, allowBatchMode = false)
public class RoiFileToCSVConverter extends MTBOperator {

	/**
	 * Identifier string for this operator.
	 */
	private static final String operatorID = "[RoiFileToCSVConverter] ";
	
	/**
	 * Input ROI file or zip archive with ROIs.
	 */
	@Parameter(label = "Input ROI file", required = true, dataIOOrder = 0, 
			direction = Direction.IN, description = "ImageJ ROI file.")
	private ALDFileString roiFile = null;

	/**
	 * Output file name.
	 * <p>
	 * In case that the input file contains multiple ROIs, the output filename
	 * is extended by a unique number and ending "*.cvs". Note that if only a
	 * single ROI is to be written and there is no or a wrong ending given this
	 * won't be corrected here.
	 */
	@Parameter(label = "Output CSV file", required = true, dataIOOrder = 1,
			direction = Direction.IN,	description = "Output CSV file.")
	private ALDFileString csvFile = null;

	/**
	 * Default constructor.
	 * @throws ALDOperatorException Thrown in case of failure.
	 */
	public RoiFileToCSVConverter() throws ALDOperatorException {
		// nothing to do here
	}
	
	/**
	 * Specify the ROI file to process.
	 * 
	 * @param file	ROI file to convert, needs to be either in roi or zip format.
	 */
	public void setROIFile(String file) {
		this.roiFile = new ALDFileString(file);
	}

	/**
	 * Specify the file where to save csv data.
	 * 
	 * @param file	Filename where to save the data.
	 */
	public void setCSVFile(String file) {
		this.csvFile = new ALDFileString(file);
	}

	@Override
	protected void operate() 
			throws ALDOperatorException, ALDProcessingDAGException {

		MTBRegion2DSet rset = 
				RoiManagerAdapter.getInstance().getRegionSetFromRoiFile(
						this.roiFile.toString());
		if (rset.size() == 1) {
			
			this.fireOperatorExecutionProgressEvent(
					new ALDOperatorExecutionProgressEvent(this, operatorID 
						+ "Processing file " + this.roiFile + " containing one region..."));

			if (!this.writeRoiToFile(rset.elementAt(0), this.csvFile.toString()))
				throw new ALDOperatorException(OperatorExceptionType.OPERATE_FAILED, 
						operatorID + "converting ROI file failed!");
		}
		else {
			
			this.fireOperatorExecutionProgressEvent(
				new ALDOperatorExecutionProgressEvent(this, operatorID 
					+ "Processing file " + this.roiFile + " containing " + rset.size() 
						+ " regions..."));

			String outRoot = 
					ALDFilePathManipulator.removeExtension(this.csvFile.toString());
			String ofile;
			int counter = 1;
			for (MTBRegion2D r: rset) {
				
				String nc = String.format("%03d", new Integer(counter));
				
				this.fireOperatorExecutionProgressEvent(
						new ALDOperatorExecutionProgressEvent(this, operatorID 
							+ "-> converting region " + nc + "..."));
				
				ofile = outRoot + "-" + nc + ".csv";
				if (!this.writeRoiToFile(r, ofile))
					System.err.println(
						operatorID + " writing ROI with ID " + counter + " failed!");
				++counter;
			}
		}

		this.fireOperatorExecutionProgressEvent(
				new ALDOperatorExecutionProgressEvent(this, operatorID 
					+ "Operations completed!"));

	}
	
	/**
	 * Writes the given region to the specified file in csv format.
	 * 
	 * @param r			Region to write.
	 * @param file	File to where the region is to be written.
	 * @return			True in case of success, false in case of failure.
	 */
	private boolean writeRoiToFile(MTBRegion2D r, String file) {
		BufferedWriter bf = null; 
		try {
			bf = new BufferedWriter(new FileWriter(file));
			MTBContour2D rc = r.getContour();
			Point2D.Double p;
			for (int i=0; i<rc.getPointNum(); ++i) {
				p = rc.getPointAt(i);
				bf.write((int)p.x + "," + (int)p.y + "\n");
			}
			bf.flush();
			bf.close();
			return true;
		} catch (IOException e) {
			try {
				if (bf != null)
					bf.close();
			} catch (IOException e1) {
				return false;
			}
			return false;
		} catch (ALDOperatorException e) {
			try {
				bf.close();
			} catch (IOException e1) {
				return false;
			}
			return false;
		} catch (ALDProcessingDAGException e) {
			try {
				bf.close();
			} catch (IOException e1) {
				return false;
			}
			return false;
		}
	}
}
